<?php

class SQLFilter {
    
    private $where = " WHERE";
    private $bindParams = [];
    private $intBindParams = [];
    private $openParenthesis = 0;
    private $canAddExpressions = true;
    private $limit = false;
    
    /**
     * Expression is in the format
     * column_name operator value
     * Don't put quotes on strings
     * e.g age > 17, phone LIKE %69% etc 
     * @param type $expression
     */
    function __construct($expression) {
        $this->addExpression($expression);
    }
    
    /**
     * Add a new AND expresion
     * expr format age > 17, phone LIKE %69% etc 
     * Don't put quotes on strings
     * @param string $expression
     * @return $this
     */
    function andExpr($expression) {
        $this->where .= " AND";
        $this->addExpression($expression);
        return $this;
    }
    
    /**
     * Add a new OR expression
     * expr format age > 17, phone LIKE %69% etc
     * Don't put quotes on strings
     * @param string $expression
     * @return $this
     */
    function orExpr($expression) {
        $this->where .= " OR";
        $this->addExpression($expression);
        return $this;
    }
    
    /**
     * Adds an left bracket to group AND expressions
     * expr format age > 17, phone LIKE %69% etc
     * Don't put quotes on strings
     * @return $this
     */
    function andLBr($expression) {
        $this->openParenthesis++;
        $this->where .= " AND (";
        $this->addExpression($expression);
        return $this;
    }
    
        /**
     * Adds an left bracket to group OR expressions
     * expr format age > 17, phone LIKE %69% etc
     * Don't put quotes on strings
     * @return $this
     */
    function orLBr($expression) {
        $this->openParenthesis++;
        $this->where .= " OR (";
        $this->addExpression($expression);
        return $this;
    }
    
     /**
     * Adds a right bracket to close group expressions
     * @return $this
     */
    function rBr() {
        $this->openParenthesis--;
        $this->where .= " )";
        return $this;
    }
    
    /**
     * limit expression format
     * limit or limit_start, limit_end
     * e.g 5 or 5,3
     * @param string $limit_expression
     */
    function limit($limit_expression) {
        $this->where .= " LIMIT";
        $this->addLimitExpression($limit_expression);
        return $this;
    }
    
    function orderBy($order_by_expr) {
        if(!$this->canAddExpressions) {
            throw new InvalidFilterException("Can't add ORDER BY expression after LIMIT or ORDER BY expressions");
        }
        if(!preg_match('/^\w+(( )+(ASC|DESC))?(( )*,( )+\w+(( )+(ASC|DESC))?)*$/', $order_by_expr)) {
            throw new InvalidArgumentException("Bad order by format!");
        }
        $this->where .= " ORDER BY $order_by_expr";
        $this->canAddExpressions = false;
        return $this;
    }
    
    /**
     * Retrieving the WHERE part of the sql
     * @return string the where expr e.g WHERE name = :name
     */
    function getFilter() {
        return $this->where;
    } 
    
    function getBindParams() {
        return $this->bindParams;
    }
    
    function getIntBindParams() {
        return $this->intBindParams;
    }
    
    /**
     * Validates the filter by checking the parenthesis
     * @throws InvalidFilterException
     */
    function validate() {
        if($this->openParenthesis != 0) {
            throw new InvalidFilterException("Parenthesis missmatched!");
        }
    }
    
    private function addExpression($expression) {
        if(!$this->canAddExpressions) {
            throw new InvalidFilterException("Can not add more expressions after LIMIT or ORDER BY expressions");
        }
        $exp_parts = explode(" ", $expression);
        if(sizeof($exp_parts) != 3) {
            throw new InvalidArgumentException("Bad filter expression given!");
        }
        $column = $exp_parts[0];
        $value = $exp_parts[2];
        $this->where .= " $column {$exp_parts[1]} :$column";
        $this->bindParams[":$column"] = $value; 
    } 
    
    private function addLimitExpression($limit_expression) {
        if($this->limit) {
            throw new InvalidFilterException("LIMIT expression already set!");
        }
        if(!preg_match('/^\d+( )*((,)( )*\d+)?$/', $limit_expression)) {
            throw new InvalidArgumentException("Wrong limit expression format");
        }
        $exp_parts = explode(",", $limit_expression);
        $count = 0;
        foreach ($exp_parts as $key=>$exp_part) {
            $exp_part = intval(trim($exp_part));
            $this->intBindParams[":lm$count"] = $exp_part; 
            $exp_parts[$key] = ":lm$count";
            $count++;
        }
        $this->where .= " ".implode(", ", $exp_parts);
        $this->limit = true;
        $this->canAddExpressions = false;
    }
    
}

/**
 * Exception for handling not valid filter
 */
class InvalidFilterException extends Exception
{
    public function __construct($message, $code = 0, Exception $previous = null) {
        parent::__construct($message, $code, $previous);
    }

    // custom string representation of object
    public function __toString() {
        return __CLASS__ . ": [{$this->code}]: {$this->message}\n";
    }
}

