<?php

/**
 * Adds CRUD functionality to all
 * child DAO classes
 */
abstract class CrudDao {

    /**
     * @var string 
     */
    private $table_name;

    /**
     * @var string 
     */
    private $class_name;

    /**
     * @var PDO 
     */
    protected $connection;

    /**
     * @param PDO $connection The pdo connction
     */
    public function __construct($connection) {
        $this->connection = $connection;
        $this->connection->exec("SET character_set_results = 'utf8', character_set_client = 'utf8', character_set_connection = 'utf8', character_set_database = 'latin1', character_set_server = 'latin1'");
        $this->table_name = $this->getTableName();
        $this->class_name = $this->getClassName();
    }

    /**
     * @return string The table name in database
     */
    abstract protected function getTableName();

    /**
     * @return RowMapper The mapper for an 
     */
    abstract protected function getClassName();

    /**
     * @param $sqlFilter SQLFilter
     * @param string $orderBy e.g DESC name, 
     * @return array() The list of objects
     */
    public function findAll($sqlFilter = null) {

        try {
            $sql = "SELECT * FROM {$this->table_name}";
            if ($sqlFilter) {
                $sqlFilter->validate();
                $sql .= $sqlFilter->getFilter();
            }
            $stmt = $this->connection->prepare($sql);
            if ($sqlFilter) {
                $this->bindSqlFilterParams($stmt, $sqlFilter);
                $stmt->execute();
            } else {
                $stmt->execute();
            }
            $objects = [];
            while ($object = $stmt->fetchObject($this->class_name, [$this->connection])) {
                $objects[] = $object;
            }
            return $objects;
        } catch (Exception $e) {
            echo "error: $e";
            die("Database Error");
        } finally {
            $stmt->closeCursor();
        }
    }

    /**
     * @return object the object or false
     */
    public function findById($id) {
        try {
            $sql = "SELECT * FROM {$this->table_name} WHERE id = :id";
            $stmt = $this->connection->prepare($sql);
            $stmt->execute([":id" => $id]);
            return $stmt->fetchObject($this->class_name);
        } catch (Exception $e) {
            die("Database Error");
        } finally {
            $stmt->closeCursor();
        }
    }

    /**
     * Inserts a new object in database or updates if not exists
     * @param Object $object
     */
    public function save($object) {

        $reflection = new ReflectionClass($object);
        $properties = $reflection->getProperties();
        if (!$this->exists($object)) {
            return $this->insert($object, $properties);
        } else {
            return $this->update($object, $properties);
        }
    }

    public function delete($object) {
        try {
            $sql = "DELETE FROM {$this->table_name} WHERE id = :id";
            $stmt = $this->connection->prepare($sql);
            $id = $object->getId();
            $stmt->execute([":id" => $id]);
            return $stmt->rowCount() > 0;
        } catch (Exception $e) {
            die("Database Error");
        } finally {
            $stmt->closeCursor();
        }
    }

    /**
     * Determines whether or not an object 
     * is persisted in database
     * @param object $object
     * @return boolean 
     */
    public function exists($object) {

        if ($this->findById($object->getId())) {
            return true;
        }
        return false;
    }

    private function insert($object, $properties) {
        try {
            $names = [];
            $values = [];
            foreach ($properties as $property) {
                $property->setAccessible(true);
                $name = $property->getName();
                $names[] = ":$name";
                $values[] = $property->getValue($object);
            }
            $sql = "INSERT INTO {$this->table_name} VALUES (" . join(", ", $names) . ")";
            $stmt = $this->connection->prepare($sql);
            for ($i = 0; $i < sizeof($names); $i++) {
                $stmt->bindParam($names[$i], $values[$i]);
            }
            $stmt->execute();
            return $stmt->rowCount() > 0;
        } catch (Exception $e) {
            die("Database error $e!");
        } finally {
            $stmt->closeCursor();
        }
    }

    private function update($object, $properties) {
        try {
            $pairs = [];
            $values = [];
            $names = [];
            foreach ($properties as $property) {
                $property->setAccessible(true);
                $name = $property->getName();
                $names[] = ":$name";
                $pairs[] = "$name = :$name";
                $values[] = $property->getValue($object);
            }
            $sql = "UPDATE {$this->table_name} SET " . join(", ", $pairs) . " WHERE id = :id";
            $stmt = $this->connection->prepare($sql);
            for ($i = 0; $i < sizeof($names); $i++) {
                $stmt->bindParam($names[$i], $values[$i]);
            }
            $stmt->execute();
            return $stmt->rowCount() > 0;
        }
        catch (Exception $e) {
            die("Database error");
        }
        finally {
            $stmt->closeCursor();
        }
    }

    private function bindSqlFilterParams($stmt, $sqlFilter) {
        foreach ($sqlFilter->getIntBindParams() as $key => &$value) {
            $stmt->bindParam($key, $value, PDO::PARAM_INT);
        }
        foreach ($sqlFilter->getBindParams() as $key => &$value) {
            $stmt->bindParam($key, $value);
        }
    }

}
