<?php
// xLoyalty – Clover Launcher + Dashboard

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/db.php';

// ---------- Helpers ----------

// Exchange OAuth code for access token (first install / re-auth)
function clover_token_exchange($code)
{
    $base = clover_base_url(); // from config.php
    $url  = $base . '/oauth/v2/token';

    $data = http_build_query(array(
        'client_id'     => CLOVER_CLIENT_ID,
        'client_secret' => CLOVER_CLIENT_SECRET,
        'code'          => $code,
        'grant_type'    => 'authorization_code'
    ));

    $opts = array(
        'http' => array(
            'method'  => 'POST',
            'header'  => "Content-Type: application/x-www-form-urlencoded\r\n",
            'content' => $data,
            'timeout' => 20
        )
    );

    $ctx   = stream_context_create($opts);
    $resp  = @file_get_contents($url, false, $ctx);
    $json  = $resp !== false ? json_decode($resp, true) : null;

    return is_array($json) ? $json : array();
}

// ---------- Read query params from Clover ----------

$code        = isset($_GET['code']) ? $_GET['code'] : null;
$merchant_id = isset($_GET['merchant_id']) ? $_GET['merchant_id'] : null;

// ---------- Handle OAuth token exchange ----------

if ($code && $merchant_id) {
    $tokens = clover_token_exchange($code);

    if (!empty($tokens['access_token'])) {
        // Store / update merchant record
        $stmt = $pdo->prepare(
            "REPLACE INTO clover_merchants (merchant_id, access_token, updated_at)
             VALUES (?, ?, NOW())"
        );
        $stmt->execute(array($merchant_id, $tokens['access_token']));
    }
}

// ---------- Load merchant info from DB ----------

$merchantRow  = null;
$connected    = false;
$xloyaltyId   = null;
$updatedAtStr = '';

if ($merchant_id) {
    $stmt = $pdo->prepare(
        "SELECT merchant_id, access_token, xloyalty_account_id, updated_at
         FROM clover_merchants
         WHERE merchant_id = ?"
    );
    $stmt->execute(array($merchant_id));
    $merchantRow = $stmt->fetch(PDO::FETCH_ASSOC);

    if ($merchantRow && !empty($merchantRow['access_token'])) {
        $connected = true;
    }

    if ($merchantRow && !empty($merchantRow['xloyalty_account_id'])) {
        $xloyaltyId = $merchantRow['xloyalty_account_id'];
    }

    if ($merchantRow && !empty($merchantRow['updated_at'])) {
        $updatedAtStr = $merchantRow['updated_at'];
    }
}

$statusLabel = $connected ? 'Connected' : 'Not connected';
$statusColor = $connected ? '#16a34a' : '#dc2626';

?>
<!doctype html>
<html>
<head>
    <meta charset="utf-8">
    <title>xLoyalty – Clover Dashboard</title>
    <meta name="viewport" content="width=device-width, initial-scale=1">

    <style>
        body {
            font-family: Arial, Helvetica, sans-serif;
            margin: 0;
            padding: 0;
            background: #f3f4f6;
            color: #111827;
        }
        .wrapper {
            max-width: 960px;
            margin: 0 auto;
            padding: 16px;
        }
        .card {
            background: #ffffff;
            border-radius: 8px;
            padding: 16px 20px;
            margin-bottom: 16px;
            box-shadow: 0 1px 3px rgba(15, 23, 42, 0.08);
        }
        h1 {
            font-size: 20px;
            margin: 0 0 8px 0;
        }
        h2 {
            font-size: 18px;
            margin: 0 0 8px 0;
        }
        .status-pill {
            display: inline-block;
            padding: 4px 10px;
            border-radius: 999px;
            font-size: 12px;
            font-weight: bold;
            color: #ffffff;
        }
        .status-row {
            margin-top: 8px;
            margin-bottom: 8px;
        }
        .label {
            font-weight: bold;
        }
        .value {
            margin-left: 4px;
        }
        .btn {
            display: inline-block;
            padding: 8px 14px;
            border-radius: 6px;
            border: none;
            background: #2563eb;
            color: #ffffff;
            font-size: 14px;
            cursor: pointer;
            text-decoration: none;
        }
        .btn-secondary {
            background: #4b5563;
        }
        .btn:disabled {
            background: #9ca3af;
            cursor: default;
        }
        .field {
            margin-bottom: 8px;
        }
        .field label {
            display: block;
            font-size: 13px;
            margin-bottom: 2px;
        }
        .field input {
            width: 100%;
            padding: 6px 8px;
            border-radius: 4px;
            border: 1px solid #d1d5db;
            font-size: 14px;
        }
        .hint {
            font-size: 12px;
            color: #6b7280;
        }
        .row {
            display: flex;
            flex-wrap: wrap;
            gap: 12px;
        }
        .col-half {
            flex: 1 1 48%;
            min-width: 200px;
        }
    </style>
</head>
<body>
<div class="wrapper">

    <div class="card">
        <h1>xLoyalty – Clover Integration</h1>
        <p class="hint">
            This screen is used to verify the connection between Clover and xLoyalty,
            and to trigger loyalty actions such as redeeming points.
        </p>
    </div>

    <div class="card">
        <h2>Connection status</h2>

        <div class="status-row">
            <span class="status-pill" style="background: <?php echo $statusColor; ?>;">
                <?php echo htmlspecialchars($statusLabel); ?>
            </span>
        </div>

        <p>
            <span class="label">Clover merchant ID:</span>
            <span class="value">
                <?php echo $merchant_id ? htmlspecialchars($merchant_id) : '<em>not provided</em>'; ?>
            </span>
        </p>

        <p>
            <span class="label">xLoyalty account ID:</span>
            <span class="value">
                <?php
                if ($xloyaltyId) {
                    echo htmlspecialchars($xloyaltyId);
                } else {
                    echo '<em>not linked yet</em>';
                }
                ?>
            </span>
        </p>

        <?php if ($updatedAtStr): ?>
            <p class="hint">
                Last updated: <?php echo htmlspecialchars($updatedAtStr); ?>
            </p>
        <?php endif; ?>

        <p style="margin-top: 12px;">
            <?php if ($merchant_id): ?>
                <a class="btn-secondary btn"
                   href="link_xloyalty.php?merchant_id=<?php echo urlencode($merchant_id); ?>">
                    Link / Change xLoyalty account
                </a>
            <?php else: ?>
                <span class="hint">Merchant ID is missing. Open this app from Clover POS to initialize the connection.</span>
            <?php endif; ?>
        </p>
    </div>

    <div class="card">
        <h2>Redeem points (test panel)</h2>
        <p class="hint">
            Use this panel to trigger a redeem flow for the current order.
            In a production setup, some of these values will be passed automatically
            from Clover (merchant, order, customer).
        </p>

        <form method="post" action="redeem_order.php">
            <div class="row">
                <div class="col-half">
                    <div class="field">
                        <label for="merchant_id">Merchant ID</label>
                        <input type="text" id="merchant_id" name="merchant_id"
                               value="<?php echo $merchant_id ? htmlspecialchars($merchant_id) : ''; ?>" />
                    </div>
                </div>
                <div class="col-half">
                    <div class="field">
                        <label for="customer_id">Customer ID (xLoyalty)</label>
                        <input type="text" id="customer_id" name="customer_id" placeholder="e.g. CLV-12345" />
                    </div>
                </div>
            </div>

            <div class="row">
                <div class="col-half">
                    <div class="field">
                        <label for="order_id">Clover Order ID</label>
                        <input type="text" id="order_id" name="order_id" placeholder="Clover order id" />
                    </div>
                </div>
                <div class="col-half">
                    <div class="field">
                        <label for="order_amount">Order amount (€)</label>
                        <input type="text" id="order_amount" name="order_amount" placeholder="e.g. 25.50" />
                    </div>
                </div>
            </div>

            <div class="field">
                <label for="notes">Notes (optional)</label>
                <input type="text" id="notes" name="notes" placeholder="Extra notes for the redemption" />
            </div>

            <p class="hint">
                When you submit this form, the system should:
                (1) call the xLoyalty redeem API, (2) get the discount value,
                and (3) apply the discount to the Clover order.
            </p>

            <button type="submit" class="btn" <?php echo $connected ? '' : 'disabled="disabled"'; ?>>
                Redeem points now
            </button>
        </form>
    </div>

</div>
</body>
</html>
